#include "stdafx.h"
#include "MappingStatus.h"

/// <summary>
/// Default constructor
/// </summary>
/// <param name="p_canId">CAN ID to map to CAN-ISO-TP network addressing information</param>
/// <param name="p_canIdResponse">CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
/// (i.e. the Flow Control frames will use this ID)</param>
/// <param name="p_canIdType">The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)</param>
/// <param name="p_formatType">The ISO-TP network addressing format.</param>
/// <param name="p_msgType">Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)</param>
/// <param name="p_sourceAddr">Source address</param>
/// <param name="p_targetAddr">Target address</param>
/// <param name="p_targetType">Type of addressing (physical: -node to node-, or functional: -node to all-)</param>
/// <param name="p_remoteAddr">Remote address (used only with remote disgnostic message)</param>
MappingStatus::MappingStatus(
	UINT32 p_canId = 0x00,
	UINT32 p_canIdResponse = 0x00,
	TPCANTPIdType p_canIdType = PCANTP_ID_CAN_11BIT,
	TPCANTPFormatType p_formatType = PCANTP_FORMAT_UNKNOWN,
	TPCANTPMessageType p_msgType = PCANTP_MESSAGE_UNKNOWN,
	TPCANTPAddressingType p_targetType = PCANTP_ADDRESSING_UNKNOWN,
	byte p_sourceAddr = 0x00,
	byte p_targetAddr = 0x00,
	byte p_remoteAddr = 0x00) {

	m_canId = p_canId;
	m_canIdResponse = p_canIdResponse;
	m_canIdType = p_canIdType;
	m_formatType = p_formatType;
	m_msgType = p_msgType;
	m_sourceAddr = p_sourceAddr;
	m_targetAddr = p_targetAddr;
	m_targetType = p_targetType;
	m_remoteAddr = p_remoteAddr;

}

/// <summary>
/// Gets a new instance of the object.
/// </summary>
/// <returns>The cloned instance of the object.</returns>
MappingStatus MappingStatus::Clone() {

	return MappingStatus(m_canId, m_canIdResponse, m_canIdType,
		m_formatType, m_msgType, m_targetType, m_sourceAddr, m_targetAddr, m_remoteAddr);

}

/// <summary>
/// The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)
/// </summary>
TPCANTPIdType MappingStatus::GetCanIdType() {
	return m_canIdType;
}

/// <summary>
/// States if an ISO-TP mapping is NOT required for this configuration.
///  Some 29 bits CAN ID ISO-TP message with specific format addressing 
///  do NOT require ISO-TP mappings.
/// </summary>
bool MappingStatus::IsAutomatic() {

	if (GetCanIdType() == PCANTP_ID_CAN_29BIT && (
		m_formatType == PCANTP_FORMAT_FIXED_NORMAL ||
		m_formatType == PCANTP_FORMAT_MIXED ||
		m_formatType == PCANTP_FORMAT_ENHANCED)
		)
		return true;
	return false;

}

/// <summary>
/// The formatted name of this mapping configuration.
/// </summary>
CString MappingStatus::Name() {

	CString name;

	name = MappingStatus::CanIdType() + L", " + MappingStatus::FormatType() + L", " + MappingStatus::TargetType();
	if (!IsAutomatic())
		name += L": " + MappingStatus::SourceAddress() + L" -> " + MappingStatus::TargetAddress();
	return name;
}

/// <summary>
/// The CAN ID of the configured mapping as a string.
/// Note: automatic mapping has no CAN ID, since it is 
/// handled on the fly by the ISO-TP API.
/// </summary>
CString MappingStatus::CanId() {
	if (IsAutomatic())
		return L"-";
	return CPCanIsoTpUtils::GetCanId(m_canId, m_canIdType == PCANTP_ID_CAN_29BIT);
}

/// <summary>
/// The CAN ID response of the configured mapping as a string.
/// Note: automatic mapping has no CAN ID response, since it is 
/// handled on the fly by the ISO-TP API.
/// </summary>
CString MappingStatus::CanIdResponse() {
	if (IsAutomatic())
		return L"-";
	return CPCanIsoTpUtils::GetCanId(m_canIdResponse, m_canIdType == PCANTP_ID_CAN_29BIT);
}

/// <summary>
/// The CAN ID type of the configured mapping as a string.
/// </summary>
CString MappingStatus::CanIdType() {
	return CPCanIsoTpUtils::GetCanIdType(m_canIdType);
}

/// <summary>
/// The ISO-TP message type of the configured mapping as a string.
/// </summary>
CString MappingStatus::MsgType() {
	return CPCanIsoTpUtils::GetMsgType(m_msgType);
}

/// <summary>
/// The ISO-TP addressing format type of the configured mapping as a string.
/// </summary>
CString MappingStatus::FormatType() {
	return CPCanIsoTpUtils::GetFormatType(m_formatType);
}

/// <summary>
/// The ISO-TP target address type ID type of the configured mapping as a string.
/// </summary>
CString MappingStatus::TargetType() {
	return CPCanIsoTpUtils::GetTargetType(m_targetType);
}

/// <summary>
/// The source address of the configured mapping as a string.
/// </summary>
CString MappingStatus::SourceAddress() {
	if (IsAutomatic())
		return L"-";
	return CPCanIsoTpUtils::GetAddress(m_sourceAddr);
}

/// <summary>
/// The target address of the configured mapping as a string.
/// </summary>
CString MappingStatus::TargetAddress() {
	if (IsAutomatic())
		return L"-";
	return CPCanIsoTpUtils::GetAddress(m_targetAddr);
}

/// <summary>
/// The remote address of the configured mapping as a string.
/// </summary>
CString MappingStatus::RemoteAddress() {
	if (IsAutomatic())
		return L"-";
	return CPCanIsoTpUtils::GetAddress(m_remoteAddr);
}